<?php
/**
 * Contact Tag Removed
 *
 * @package     AutomatorWP\Integrations\Jetpack_CRM\Triggers\Contact_Tag_Removed
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Jetpack_CRM_Contact_Tag_Removed extends AutomatorWP_Integration_Trigger {

    public $integration = 'jetpack_crm';
    public $trigger = 'jetpack_crm_contact_tag_removed';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'Tag removed from contact', 'automatorwp-pro' ),
            'select_option'     => __( '<strong>Tag</strong> removed from contact', 'automatorwp-pro' ),
            /* translators: %1$s: Tag. %2$s: Number of times. */
            'edit_label'        => sprintf( __( '%1$s removed from contact %2$s time(s)', 'automatorwp-pro' ), '{tag}', '{times}' ),
            /* translators: %1$s: Tag. */
            'log_label'         => sprintf( __( '%1$s removed from contact', 'automatorwp-pro' ), '{tag}' ),
            'action'            => 'zbs_tag_removed_from_objid',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'tag' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'tag',
                    'name'              => __( 'Tag:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any tag', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_jetpack_crm_get_contact_tags',
                    'options_cb'        => 'automatorwp_jetpack_crm_options_cb_tag',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_jetpack_crm_get_contact_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $tagID
     * @param int $objtype
     * @param int $objid
     */
    public function listener( $tagID, $objtype, $objid ) {

        $user_id = get_current_user_id();

        // Make sure subscriber has a user ID assigned
        if ( $user_id === 0 ) {
            return;
        }

        // To get the contact data
        $contact_data = zeroBS_getCustomer( $objid );

        // Trigger the tag removed
        automatorwp_trigger_event( array(
            'trigger'               => $this->trigger,
            'user_id'               => $user_id,
            'tag_id'                => $tagID,
            'contact_first_name'    => $contact_data['fname'],
            'contact_last_name'     => $contact_data['lname'],
            'contact_email'         => $contact_data['email']
        ) );
        
    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['tag_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( $trigger_options['tag'] !== 'any' && absint( $trigger_options['tag'] ) !== absint( $event['tag_id'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }    
  
    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['contact_first_name'] = ( isset( $event['contact_first_name'] ) ? $event['contact_first_name'] : '' );
        $log_meta['contact_last_name'] = ( isset( $event['contact_last_name'] ) ? $event['contact_last_name'] : '' );
        $log_meta['contact_email'] = ( isset( $event['contact_email'] ) ? $event['contact_email'] : '' );

        return $log_meta;

    }
    
}

new AutomatorWP_Jetpack_CRM_Contact_Tag_Removed();